;;; Waxeye Parser Generator
;;; www.waxeye.org
;;; Copyright (C) 2008 Orlando D. A. R. Hill
;;;
;;; Permission is hereby granted, free of charge, to any person obtaining a copy of
;;; this software and associated documentation files (the "Software"), to deal in
;;; the Software without restriction, including without limitation the rights to
;;; use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
;;; of the Software, and to permit persons to whom the Software is furnished to do
;;; so, subject to the following conditions:
;;;
;;; The above copyright notice and this permission notice shall be included in all
;;; copies or substantial portions of the Software.
;;;
;;; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
;;; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
;;; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
;;; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
;;; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
;;; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
;;; SOFTWARE.


(module
java
mzscheme

(require (lib "ast.ss" "waxeye")
         (lib "fa.ss" "waxeye")
         (only (lib "list.ss" "mzlib") filter)
         "code.scm" "dfa.scm" "gen.scm" "util.scm")
(provide gen-java)


(define *java-parser-name* "")
(define *java-node-name* "")
(define *java-tree-type* "")


(define (java-comment lines)
  (comment-bookend "/*" " *" " */" lines))


(define (java-doc . lines)
  (comment-bookend "/**" " *" " */" lines))


(define (java-header-comment)
  (if *file-header*
      (java-comment *file-header*)
      (java-comment *default-header*)))


(define (gen-java-names)
  (set! *java-node-name* (if *name-prefix*
                             (string-append *name-prefix* "Type")
                             "Type"))
  (set! *java-parser-name* (if *name-prefix*
                               (string-append *name-prefix* "Parser")
                               "Parser"))
  (set! *java-tree-type* (string-append "IAST<" *java-node-name* ">")))


(define (gen-java grammar path)
  (gen-java-names)
  (dump-string (java-type grammar) (string-append path *java-node-name* ".java"))
  (dump-string (java-parser grammar) (string-append path *java-parser-name* ".java")))


(define (java-type grammar)
  (let ((non-terms (get-non-terms grammar)))
    (format "~a~a\n~apublic enum ~a\n{\n~a}\n"
            (java-header-comment)
            (gen-java-package)
            (java-doc "The types of AST nodes." "" "@author Waxeye Parser Generator")
            *java-node-name*
            (indent (string-append
                     (ind) "_Empty,\n"
                     (ind) "_Char,\n"
                     (ind) "_Pos,\n"
                     (ind) "_Neg"
                     (string-concat (map (lambda (a)
                                           (format ",\n~a~a"
                                                   (ind)
                                                   (camel-case-upper a)))
                                         non-terms))
                     "\n")))))


(define (java-parser grammar)
  (format "~a~a\n~a~apublic final class ~a extends org.waxeye.parser.Parser<~a>\n{\n~a}\n"
          (java-header-comment)
          (gen-java-package)
          (gen-java-imports)
          (java-doc "A parser generated by the Waxeye Parser Generator." "" "@author Waxeye Parser Generator")
          *java-parser-name*
          *java-node-name*
          (indent (string-append (gen-constructor) "\n" (gen-make-automata (make-automata grammar))))))


(define (gen-java-package)
  (if *module-name*
      (format "package ~a;\n" *module-name*)
      ""))


(define (gen-java-imports)
  "import java.util.ArrayList;
import java.util.List;

import org.waxeye.parser.AutomatonTransition;
import org.waxeye.parser.CharTransition;
import org.waxeye.parser.Edge;
import org.waxeye.parser.FA;
import org.waxeye.parser.State;
import org.waxeye.parser.WildCardTransition;

")


(define (gen-constructor)
  (format "~a~apublic ~a()\n~a{\n~a~a}\n"
          (java-doc (format "Creates a new ~a." *java-parser-name*))
          (ind)
          *java-parser-name*
          (ind)
          (indent
           (format "~asuper(makeAutomata(), true, ~a, ~a._Empty, ~a._Char, ~a._Pos, ~a._Neg);\n"
                   (ind)
                   *start-index*
                   *java-node-name*
                   *java-node-name*
                   *java-node-name*
                   *java-node-name*))
          (ind)))


(define (gen-make-automata automata)
  (format "~a~aprivate static List<FA<~a>> makeAutomata()\n~a{\n~a~a}\n"
          (java-doc "Builds the automata for the parser." "" "@return The automata for the parser.")
          (ind)
          *java-node-name*
          (ind)
          (indent
           (string-append
            (format "~aList<Edge<~a>> edges;\n" (ind) *java-node-name*)
            (format "~aList<State<~a>> states;\n" (ind) *java-node-name*)
            (format "~afinal List<FA<~a>> automata = new ArrayList<FA<~a>>();\n" (ind) *java-node-name* *java-node-name*)
            "\n"
            (string-concat (map gen-fa (vector->list automata)))
            (string-append (ind) "return automata;\n")))
          (ind)))


(define (gen-fa a)
  (format "~astates = new ArrayList<State<~a>>();\n~a~a"
          (ind)
          *java-node-name*
          (string-concat (map gen-state (vector->list (fa-states a))))
          (format "~aautomata.add(new FA<~a>(~a.~a, ~a, states));\n\n"
                  (ind)
                  *java-node-name*
                  *java-node-name*
                  (let ((type (fa-type a)))
                    (cond
                     ((equal? type '&) "_Pos")
                     ((equal? type '!) "_Neg")
                     (else
                      (camel-case-upper (symbol->string type)))))
                  (case (fa-mode a)
                    ((voidArrow) "FA.VOID")
                    ((pruneArrow) "FA.PRUNE")
                    ((leftArrow) "FA.LEFT")))))


(define (gen-state s)
  (format "~aedges = new ArrayList<Edge<~a>>();\n~a~astates.add(new State<~a>(edges, ~a));\n"
          (ind)
          *java-node-name*
          (string-concat (map gen-edge (state-edges s)))
          (ind)
          *java-node-name*
          (bool->s (state-match s))))


(define (gen-edge e)
  (format "~aedges.add(new Edge<~a>(new ~a, ~a, ~a));\n"
          (ind)
          *java-node-name*
          (gen-trans (edge-t e))
          (edge-s e)
          (bool->s (edge-v e))))


(define (gen-trans t)
  (cond
   ((equal? t 'wild) (gen-wild-card-trans))
   ((integer? t) (gen-automaton-trans t))
   ((char? t) (gen-char-trans t))
   ((pair? t) (gen-char-class-trans t))))


(define (gen-automaton-trans t)
  (format "AutomatonTransition<~a>(~a)" *java-node-name* t))


(define (gen-char-trans t)
  (format "CharTransition<~a>(new char[]{~a}, new char[]{}, new char[]{})" *java-node-name* (gen-char t)))


(define (gen-char-class-trans t)
  (let* ((single (filter char? t))
         (ranges (filter pair? t))
         (min (map car ranges))
         (max (map cdr ranges)))
    (format "CharTransition<~a>(~a, ~a, ~a)"
            *java-node-name*
            (gen-char-list single)
            (gen-char-list min)
            (gen-char-list max))))


(define (gen-char-list l)
  (format "new char[]{~a}"
          (if (null? l)
              ""
              (string-append
               (gen-char (car l))
               (string-concat (map (lambda (a)
                                     (string-append ", " (gen-char a)))
                                   (cdr l)))))))


(define (gen-char t)
  (format "'~a~a'"
          (if (escape-for-java-char? t) "\\" "")
          (cond
           ((equal? t #\linefeed) "\\n")
           ((equal? t #\tab) "\\t")
           ((equal? t #\return) "\\r")
           (else t))))


(define (gen-wild-card-trans)
  (format "WildCardTransition<~a>()" *java-node-name*))

)
