#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>
#include <mmsystem.h>

#include "../ludum.h"
#include "micromod.h"

#define SAMPLING_FREQ  48000  /* 48khz. */
#define OVERSAMPLE     2      /* 2x oversampling. */
#define NUM_CHANNELS   2      /* Stereo. */
#define BUFFER_SAMPLES 4096   /* 16k per buffer. */
#define NUM_BUFFERS    8      /* 4 buffers (256k). */

static HANDLE semaphore;
static WAVEHDR wave_headers[ NUM_BUFFERS ];
static short mix_buffer[ BUFFER_SAMPLES * NUM_CHANNELS * OVERSAMPLE ];
static short wave_buffers[ NUM_BUFFERS ][ BUFFER_SAMPLES * NUM_CHANNELS ];
static long filt_l, filt_r;
static WAVEFORMATEX wave_format;
static HWAVEOUT h_wave_out;

HANDLE musicThreadHandle;
volatile int musicNext;
volatile int musicStop;

static void __stdcall wave_out_proc( HDRVR hdrvr, UINT uMsg, DWORD dwUser, DWORD dw1, DWORD dw2 ) {
	if( uMsg == WOM_DONE ) ReleaseSemaphore( semaphore, 1, NULL );
}

static void check_mmsys_error( int error ) {
	TCHAR str[ 64 ];
	if( error != MMSYSERR_NOERROR ) {
		waveOutGetErrorText( error, &str[0], 64 );
		MessageBox( NULL, str, "Error", MB_OK|MB_ICONERROR );
		ExitProcess( 1 );
	}
}

/*
	2:1 downsampling with simple but effective anti-aliasing.
	Count is the number of stereo samples to process, and must be even.
	input may point to the same buffer as output.
*/
static void downsample( short *input, short *output, long count ) {
	long in_idx, out_idx, out_l, out_r;
	in_idx = out_idx = 0;
	while( out_idx < count ) {	
		out_l = filt_l + ( input[ in_idx++ ] >> 1 );
		out_r = filt_r + ( input[ in_idx++ ] >> 1 );
		filt_l = input[ in_idx++ ] >> 2;
		filt_r = input[ in_idx++ ] >> 2;
		output[ out_idx++ ] = (short)( out_l + filt_l );
		output[ out_idx++ ] = (short)( out_r + filt_r );
	}
}

DWORD WINAPI musicThread( void *param )
{
	/* Play through once. */
	short *buffer;
	WAVEHDR *header;
	int current_buffer = 0;
	long count, samples_remaining;

	samples_remaining = micromod_calculate_song_duration();

	while(/* samples_remaining > 0 &&*/ !musicStop )
	{
		if ( musicNext != -1 )
		{
			micromod_set_position( musicNext );
			musicNext = -1;
		}

		/* Wait for a buffer to become available. */
		WaitForSingleObject( semaphore, INFINITE );

		header = &wave_headers[ current_buffer ];
		buffer = &wave_buffers[ current_buffer ][ 0 ];
		
		/* Clear mix buffer and get audio from replay. */
		count = BUFFER_SAMPLES * OVERSAMPLE;
		//if( count > samples_remaining ) count = samples_remaining;
		memset( mix_buffer, 0, BUFFER_SAMPLES * NUM_CHANNELS * OVERSAMPLE * sizeof( short ) );
		micromod_get_audio( mix_buffer, count );
		downsample( mix_buffer, buffer, BUFFER_SAMPLES * OVERSAMPLE );
		samples_remaining -= count;
		
		/* Submit buffer to audio system. */
		check_mmsys_error( waveOutUnprepareHeader( h_wave_out, header, sizeof( WAVEHDR ) ) );
		check_mmsys_error( waveOutPrepareHeader( h_wave_out, header, sizeof( WAVEHDR ) ) );
		check_mmsys_error( waveOutWrite( h_wave_out, header, sizeof( WAVEHDR ) ) );
		
		/* Next buffer. */
		current_buffer++;
		if( current_buffer >= NUM_BUFFERS ) current_buffer = 0;
	}

	return 0;
}

void musicStartup( const u8 *data )
{
	long idx;

	micromod_initialise( (signed char *)data, SAMPLING_FREQ * OVERSAMPLE );
	
	/* Initialise Wave Format Structure. */
	wave_format.wFormatTag = WAVE_FORMAT_PCM;
	wave_format.nChannels = NUM_CHANNELS;
	wave_format.nSamplesPerSec = SAMPLING_FREQ;
	wave_format.nAvgBytesPerSec = SAMPLING_FREQ * NUM_CHANNELS * 2;
	wave_format.nBlockAlign = NUM_CHANNELS * 2;
	wave_format.wBitsPerSample = 16;
	
	/* Initialise Waveform Buffers. */
	for( idx = 0; idx < NUM_BUFFERS; idx++ ) {
		WAVEHDR *header = &wave_headers[ idx ];
		memset( header, 0, sizeof( WAVEHDR ) );
		header->lpData = ( LPSTR ) &wave_buffers[ idx ][ 0 ];
		header->dwBufferLength = BUFFER_SAMPLES * NUM_CHANNELS * sizeof( short );
	}

	/* Initialise Semaphore. */
	semaphore = CreateSemaphore( NULL, NUM_BUFFERS, NUM_BUFFERS, "" );

	/* Open Audio Device. */
	check_mmsys_error( waveOutOpen( &h_wave_out, WAVE_MAPPER, &wave_format, ( DWORD ) wave_out_proc, 0, CALLBACK_FUNCTION ) );

	musicStop = false;
	musicNext = -1;
	musicThreadHandle = CreateThread( NULL, 0, musicThread, NULL, 0, NULL );
}

void musicShutdown()
{
	musicStop = true;

	WaitForSingleObject( musicThreadHandle, INFINITE );
	CloseHandle( musicThreadHandle );

	waveOutReset( h_wave_out );
	while( waveOutClose( h_wave_out ) == WAVERR_STILLPLAYING )
		Sleep( 100 );
}
