#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>
#include <d3d9.h>

#include "ludum.h"
#include "resource.h"

#include "../data/font.h"
#include "../data/cursor.h"
#include "../data/upscale_ps.h"
#include "../data/upscale_vs.h"
#include "../data/music.h"

#define PIXEL_SCALE_X		3.0f
#define PIXEL_SCALE_Y		3.6f

HWND mainWnd;
IDirect3D9 *hwD3D;
IDirect3DDevice9 *hwD3DDevice;
D3DPRESENT_PARAMETERS hwD3DParams;
IDirect3DTexture9 *hwD3DSysTex, *hwD3DVidTex;
IDirect3DVertexShader9 *hwUpscaleVS;
IDirect3DPixelShader9 *hwUpscalePS;

LBitmap *hwScreen;
LBitmap *hwFont;
LBitmap *hwCursor;
u32 hwPalette[256]; // ARGB
int letterX[256];
int letterY[256];
int letterW[256];

int mouseX, mouseY;
bool mouseDown;

LRESULT CALLBACK WndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam );
void hwStartup();
void hwDisplay( HDC hDC, HWND hWnd );
void hwGetMouse();

int ftoi( double f )
{
	int i;
	__asm {
		fld f
		fistp i
	}
	return i;
}

extern "C" void WinMainCRTStartup()
{
	MSG msg;
	WNDCLASSEX wcex;

	HINSTANCE hInstance = GetModuleHandle( NULL );

	// Register a window class.
	ZeroMemory( &wcex, sizeof(wcex) );
	wcex.cbSize = sizeof(WNDCLASSEX);
	wcex.style			= CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wcex.lpfnWndProc	= WndProc;
	wcex.cbClsExtra		= 0;
	wcex.cbWndExtra		= 0;
	wcex.hInstance		= hInstance;
	wcex.hIcon			= LoadIcon(hInstance, MAKEINTRESOURCE(IDI_LUDUM));
	wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground	= NULL;
	wcex.lpszMenuName	= NULL;
	wcex.lpszClassName	= "Ludum";
	RegisterClassEx( &wcex );

	ShowCursor( FALSE );

	DWORD dwStyle = WS_OVERLAPPEDWINDOW;
	RECT rc = { 0, 0, (int)(SCREEN_WIDTH*PIXEL_SCALE_X), (int)(SCREEN_HEIGHT*PIXEL_SCALE_Y) };
	AdjustWindowRect( &rc, dwStyle, FALSE );

	hwScreen = lbmpNew( SCREEN_WIDTH, SCREEN_HEIGHT );
	lbmpClear( hwScreen, COLOR_BLACK );
	memset( hwPalette, 0, sizeof(hwPalette) );

	// Get the font
	anmDecode( 1, incbin_font );
	hwFont = anmGetBitmap( 200, 30 );

	// Get the cursor
	anmDecode( 1, incbin_cursor );
	hwCursor = anmGetBitmap( 13, 13 );

	// Scan the font widths
	letterW[' '] = 6;
	uint letter = '!';
	for (uint row=0;row<3;row++)
	{
		uint start = 0;
		u8 *line = &hwFont->pixels[hwFont->w*(row*9+8)];
		uint w = 0;
		while( true )
		{
			while( line[start+w] == 0 )
				w++;

			letterX[letter] = start;
			letterY[letter] = row*9;
			letterW[letter] = w;

			if ( letter == '@' || letter == '_' || letter == '~' )
			{
				letter++;
				break;
			}

			letter++;
			start += w;
			w = 1;
		}
	}
	
	HWND hWnd = CreateWindowA( "Ludum", "Ludum 21", dwStyle, CW_USEDEFAULT, 0, rc.right-rc.left, rc.bottom-rc.top, NULL, NULL, hInstance, NULL );
	DWORD err = GetLastError();
	if ( !hWnd )
		ExitProcess( 1 );

	mainWnd = hWnd;

	// Initialize D3D
	hwD3D = Direct3DCreate9( D3D_SDK_VERSION );
	memset( &hwD3DParams, 0, sizeof(hwD3DParams) );
	hwD3DParams.Windowed				= TRUE;
	hwD3DParams.SwapEffect				= D3DSWAPEFFECT_DISCARD;
	hwD3DParams.BackBufferFormat		= D3DFMT_A8R8G8B8;
	hwD3DParams.EnableAutoDepthStencil	= FALSE;
	hwD3DParams.PresentationInterval	= D3DPRESENT_INTERVAL_IMMEDIATE;
	hwD3DParams.Flags					= 0;
	hwD3DParams.BackBufferWidth			= (int)(SCREEN_WIDTH*PIXEL_SCALE_X);
	hwD3DParams.BackBufferHeight		= (int)(SCREEN_HEIGHT*PIXEL_SCALE_Y);

	HRESULT hr = hwD3D->CreateDevice( D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, hWnd,
		D3DCREATE_HARDWARE_VERTEXPROCESSING|D3DCREATE_PUREDEVICE,
		&hwD3DParams, &hwD3DDevice );

	if ( FAILED(hr) )
	{
		MessageBox( hWnd, "Error creating Direct3D device\n", "Error", MB_OK|MB_ICONERROR );
		ExitProcess( 1 );
	}

	hwD3DDevice->CreateTexture( SCREEN_WIDTH, SCREEN_HEIGHT, 1, D3DUSAGE_DYNAMIC, D3DFMT_A8R8G8B8, 
		D3DPOOL_SYSTEMMEM, &hwD3DSysTex, NULL );

	hwD3DDevice->CreateTexture( SCREEN_WIDTH, SCREEN_HEIGHT, 1, 0, D3DFMT_A8R8G8B8, 
		D3DPOOL_DEFAULT, &hwD3DVidTex, NULL );

	hwD3DDevice->CreateVertexShader( (DWORD *)upscale_vs_code, &hwUpscaleVS );
	hwD3DDevice->CreatePixelShader( (DWORD *)upscale_ps_code, &hwUpscalePS );

	// Center it.
	GetWindowRect( hWnd, &rc );
	int desktopW = GetSystemMetrics( SM_CXSCREEN );
	int desktopH = GetSystemMetrics( SM_CYSCREEN );
	int windowW = rc.right - rc.left;
	int windowH = rc.bottom - rc.top;
	rc.left = ( desktopW - windowW ) / 2;
	rc.top = ( desktopH - windowH ) / 2;
	MoveWindow( hWnd, rc.left, rc.top, windowW, windowH, FALSE );	

	ShowWindow( hWnd, SW_SHOWDEFAULT );
	UpdateWindow( hWnd );

	hwStartup();
#ifdef USE_MUSIC
	musicStartup( incbin_music );
#endif

	LARGE_INTEGER prev, freq;
	QueryPerformanceCounter( &prev );
	QueryPerformanceFrequency( &freq );
	double acc = 0;

	while( true )
	{
		if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE ) )
		{
			if ( msg.message == WM_QUIT )
				break;

			TranslateMessage(&msg);
			DispatchMessage(&msg);
		} else {
			LARGE_INTEGER now;
			QueryPerformanceCounter( &now );
			LONGLONG d = now.QuadPart - prev.QuadPart;
			double diff = 60 * (d / (double)freq.QuadPart);
			prev = now;
			acc += diff;
			if ( acc > 10 )
				acc = 10;

			int numTicks = ftoi( acc );

			if ( numTicks > 0 )
			{
				acc -= numTicks;

				hwGetMouse();
				while( numTicks-- )
					gameUpdate();

				// Draw the cursor on top.
				lbmpBlitMasked( hwScreen, hwCursor, mouseX-6, mouseY-6, 0, 0, 13, 13 );
				
				mouseDown = false;

				HDC hDC = GetDC( hWnd );
				hwDisplay( hDC, hWnd );
				ReleaseDC( hWnd, hDC );
			} else {
				Sleep( 1 );
			}
		}
	}

	musicShutdown();
	ExitProcess( 0 );
}

void hwGetMouse()
{
	POINT pt;
	GetCursorPos( &pt );
	ScreenToClient( mainWnd, &pt );
	mouseX = ftoi( pt.x / PIXEL_SCALE_X );
	mouseY = ftoi( pt.y / PIXEL_SCALE_Y );
}

void fontPrint2( int x, int y, u8 c, const char *text, uint count )
{
	while( count-- )
	{
		u8 letter = *text++;
		int w = letterW[letter];

		if ( letter == ' ' )
		{
			x += w;
			continue;
		}

		int sx = letterX[letter];
		int sy = letterY[letter];

		lbmpBlitReplace( hwScreen, hwFont, x, y, sx, sy, w, 8, c );

		x += w;
	}
}

void fontPrint( int x, int y, u8 c, const char *text )
{
	int numLines = 0;
	int lineW[16];
	int counts[16];
	const char *lines[16];
	const char *p = text;
	int w = 0;
	lines[0] = p;
	while( true )
	{
		if ( *p == '\n' || *p == 0 )
		{
			counts[numLines] = p - lines[numLines];
			lineW[numLines] = w;
			numLines++;
			w = 0;
			if ( *p == 0 )
				break;
			lines[numLines] = p;
		} else {
			w += letterW[*p];
		}
		p++;
	}

	for (int n=0;n<numLines;n++)
	{	
		p = lines[n];
		uint count = counts[n];

		int dx = x - lineW[n] / 2;
		if ( dx < 5 )
			dx = 5;
		if ( dx+lineW[n] > 315 )
			dx = 315 - lineW[n];

		fontPrint2( dx-1, y, 16, p, count );
		fontPrint2( dx+1, y, 16, p, count );
		fontPrint2( dx, y-1, 16, p, count );
		fontPrint2( dx, y+1, 16, p, count );
		fontPrint2( dx, y, c, p, count );

		y += 10;
	}
}

void hwStartup()
{
	mouseDown = false;

	hwGetMouse();

	gameStartup();
}

void hwDisplay( HDC hDC, HWND hWnd )
{
	hwD3DDevice->BeginScene();

	D3DLOCKED_RECT rect;
	hwD3DSysTex->LockRect( 0, &rect, NULL, D3DLOCK_DISCARD );

	// Convert palettized out to BGRA 32-bit.
	u8 *src = hwScreen->pixels;
	for (uint y=0;y<SCREEN_HEIGHT;y++)
	{
		u32 *dest = (u32 *)( (u8 *)rect.pBits + rect.Pitch*y );
		for (uint x=0;x<SCREEN_WIDTH;x++)
		{
			u32 c = hwPalette[*src++];
			*dest++ = c;
		}
	}

	hwD3DSysTex->UnlockRect( 0 );

	// Update that into somewhere useful.
	hwD3DDevice->UpdateTexture( hwD3DSysTex, hwD3DVidTex );

	// Set up our upscale shader.
	hwD3DDevice->SetVertexShader( hwUpscaleVS );
	hwD3DDevice->SetPixelShader( hwUpscalePS );
	hwD3DDevice->SetFVF( D3DFVF_XYZ|D3DFVF_TEX1 );

	hwD3DDevice->SetTexture( 0, hwD3DVidTex );
	hwD3DDevice->SetSamplerState( 0, D3DSAMP_ADDRESSU, D3DTADDRESS_CLAMP );
	hwD3DDevice->SetSamplerState( 0, D3DSAMP_ADDRESSV, D3DTADDRESS_CLAMP );
	hwD3DDevice->SetSamplerState( 0, D3DSAMP_MINFILTER, D3DTEXF_POINT );
	hwD3DDevice->SetSamplerState( 0, D3DSAMP_MAGFILTER, D3DTEXF_POINT );

	// Blit the final image to the screen.
	float tri[2*3*5] = {
		-1, 1, 0,		0, 0,
		 1, 1, 0,		1, 0,
		-1,-1, 0,		0, 1,

		 1, 1, 0,		1, 0,
		 1,-1, 0,		1, 1,
		-1,-1, 0,		0, 1,
	};
	hwD3DDevice->DrawPrimitiveUP( D3DPT_TRIANGLELIST, 2, &tri, 5*4 );

	hwD3DDevice->EndScene();
	hwD3DDevice->Present( NULL, NULL, NULL, NULL );
}


LRESULT CALLBACK WndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	PAINTSTRUCT ps;
	HDC hDC;

	switch( message )
	{
	case WM_PAINT:
		hDC = BeginPaint( hWnd, &ps );
		hwDisplay( hDC, hWnd );
		EndPaint( hWnd, &ps );
		break;
	case WM_DESTROY:
		PostQuitMessage( 0 );
		break;
	case WM_LBUTTONDOWN:
		mouseDown = true;
		break;
	default:
		return DefWindowProc( hWnd, message, wParam, lParam );
	}
	return 0;
}
