#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>

#include "ludum.h"

#include "../data/scenes.h"

#define MAX_THREADS		1000
#define MAX_ACTORS		100
#define MAX_INVENTORY	100

uint gameCurThread = 0;
uint gameCurScene = 0;
uint gameNextScene = 0;
uint gamePrevScene = 0;
void *gameThreads[MAX_THREADS];
bool gameLocal[MAX_THREADS];

Actor gameActors[MAX_ACTORS];
Got gameGot[MAX_INVENTORY];

u8 gamePathing[16][16];
uint gamePathDist[16][16];
int gamePathJoinX[16][16];
int gamePathJoinY[16][16];
int gamePathJoinN[16][16];

LBitmap *gameSceneBg;
LBitmap *gameSceneFg;
LBitmap *gameSceneMask;
uint gameTick;

bool gameUiVisible;
uint gameClickTime;
char gameCommandLine[256];
uint gameInvScroll = 0;

Command command;
Verb gameVerb;
Noun gameHoverNoun;
bool gameHoldCommand;

uint convNumSaid = 0;
u32 convSaid[10000];
uint convNumOptions = 0;
const char *convOptions[100];
int convSelected = -1;

const char *verbs[] = {
	"Walk to",
	"Look at",
	"Talk to",
	"Use",
	"Pick up",
	"Give",
};

void gameStartup()
{
	anmGetPalette( incbin_scenes, hwPalette );

	gameCurThread = 0;
	gameCurScene = gameNextScene = gamePrevScene = 0;

	gameVerb = VERB_WALK;
	gameSceneBg = NULL;
	gameSceneFg = NULL;
	gameSceneMask = NULL;
	gameTick = 0;
	gameClickTime = 0;
	gameUiVisible = false;
	gameCommandLine[0] = 0;

	convNumSaid = 0;

	gameHoverNoun.actor = NULL;
	gameHoverNoun.got = NULL;

	for (uint n=0;n<MAX_THREADS;n++)
		gameThreads[n] = NULL;

	for (uint n=0;n<MAX_INVENTORY;n++)
		gameGot[n].name = NULL;

	gameThreads[0] = ConvertThreadToFiber( NULL );
	gameLocal[0] = false;

	scriptStartup();
}

void *scriptNew( ScriptThread *fn, void *data, bool local )
{
	int n;
	for (n=0;n<MAX_THREADS;n++)
	{
		if ( gameThreads[n] == NULL )
			break;
	}

	if ( n == MAX_THREADS )
		__debugbreak();

	void *thread = CreateFiber( 0x4000, fn, data );
	gameThreads[n] = thread;
	gameLocal[n] = local;
	return thread;
}

void scriptSetScene( uint scene )
{
	gameNextScene = scene;
}

void gameSchedule()
{
	do {
		gameCurThread++;
		if ( gameCurThread >= MAX_THREADS )
			gameCurThread = 0;
	} while( gameThreads[gameCurThread] == NULL );

	SwitchToFiber( gameThreads[gameCurThread] );
}

LBitmap *findBestFrame( Costume *costume, Anim anim, Dir dir, uint cycle )
{
	CostumeTable *table = &costume->table[0];
	while( table->frame )
	{
		if ( table->anim == anim && table->dir == dir )
			break;

		table++;
	}

	if ( !table->frame )
		return NULL;

	int frame = table->frame;
	bool flipped = false;
	if ( frame < 0 )
	{
		flipped = true;
		frame = -frame;
	}

	uint subIndex = gameTick/10 + cycle;
	switch( anim )
	{
	case ANIM_IDLE: subIndex = 0; break;
	case ANIM_USE: subIndex = 0; break;
	case ANIM_WALK: subIndex %= 4; if ( subIndex == 3 ) subIndex = 1; break;
	case ANIM_TALK: subIndex %= 2; break;
	}

	frame += subIndex;

	if ( costume->frames[flipped][frame] )
		return costume->frames[flipped][frame] ;

	// Build the image and cache it.
	for (int n=1;n<=frame;n++)
		anmDecode( n, costume->incbin );

	LBitmap *bmp = anmGetBitmap( costume->sprW, costume->sprH );
	costume->frames[flipped][frame] = bmp;

	if ( flipped )
	{
		for (uint y=0;y<bmp->h;y++)
		{
			uint w = bmp->w;
			u8 *ptr0 = &bmp->pixels[y*w];
			u8 *ptr1 = ptr0 + w - 1;
			w = ( w + 1 ) >> 1;
			while( w-- )
			{
				u8 a = *ptr0;
				u8 b = *ptr1;
				*ptr0++ = b;
				*ptr1-- = a;
			}
		}
	}

	return bmp;
}

u8 mask( int x, int y )
{
	if ( x < 0 || y < 0 || x >= (int)gameSceneMask->w || y >= (int)gameSceneMask->h )
		return 0;

	return gameSceneMask->pixels[y*gameSceneMask->w+x];
}

bool solid( int x, int y )
{
	return mask( x, y ) == 0;
}

void actorUpdateMove( Actor *actor, int dx, int dy )
{
	if ( dx == 0 && dy == 0 )
		return;

	if ( solid( actor->x+dx, actor->y+dy ) )
	{
		// Try a Y-only move, then an X-only.
		if ( !solid( actor->x, actor->y+dy ) )
		{
			dx = 0;
		} else if ( !solid( actor->x+dx, actor->y ) ) {
			dy = 0;
		}
	}

	if ( solid( actor->x+dx, actor->y+dy ) )
	{
		// Can't move, stop here.
		stop( actor );
		return;
	}

	if ( dx == 0 && dy == 0 )
	{
		actor->blockedTicks++;
	} else {
		actor->x += dx;
		actor->y += dy;
		actor->blockedTicks = 0;
	}

	if ( actor->blockedTicks > 5 && actor->anim == ANIM_WALK )
	{
		stop( actor );
	}
}

void getBestDir( int x, int y, int tx, int ty, int &dx, int &dy )
{
	u8 c0 = mask( x, y );
	u8 c1 = mask( tx, ty );

	dx = dy = 0;
	if ( c0 && c1 && c0 != c1 )
	{
		do {
			c1 = gamePathing[c0][c1];
		} while( c1 && gamePathDist[c0][c1] != 1 );

		if ( c1 )
		{
			tx = gamePathJoinX[c0][c1];
			ty = gamePathJoinY[c0][c1];
		}
	}
	
	if ( x < tx )
		dx = 1;
	if ( x > tx )
		dx = -1;
	if ( y < ty )
		dy = 1;
	if ( y > ty )
		dy = -1;
}

void actorUpdate( Actor *actor )
{
	int dx, dy;
	getBestDir( actor->x, actor->y, actor->tx, actor->ty, dx, dy );

	if ( dx != 0 || dy != 0 )
	{
		if ( dy < 0 )
			actor->dir = DIR_UP;
		if ( dy > 0 )
			actor->dir = DIR_DOWN;
		if ( dx < 0 )
			actor->dir = DIR_LEFT;
		if ( dx > 0 )
			actor->dir = DIR_RIGHT;
	}

	if ( dx == 0 && dy == 0 && actor->anim == ANIM_WALK )
		stop( actor );

	if ( ( gameTick % 8 ) == 0 )
		dx = 0;

	if ( ( gameTick % 3 ) > 0 )
		dy = 0;

	actorUpdateMove( actor, dx, dy );
}

const char *getNounName( Noun noun )
{
	if ( noun.actor )
		return noun.actor->name;
	if ( noun.got )
		return noun.got->name;
	return NULL;
}

void gameUpdate()
{
	gameTick++;
	gameClickTime++;

	if ( gameNextScene != gameCurScene )
	{
		// Kill off any local threads.
		for (uint n=0;n<MAX_THREADS;n++)
		{
			if ( gameThreads[n] && gameLocal[n] )
			{
				DeleteFiber( gameThreads[n] );
				gameThreads[n] = NULL;
			}
		}

		// Clear out all the actors
		for (uint n=0;n<MAX_ACTORS;n++)
			gameActors[n].valid = NULL;

		if ( gameSceneBg )
			lbmpFree( gameSceneBg );
		if ( gameSceneFg )
			lbmpFree( gameSceneFg );
		if ( gameSceneMask )
			lbmpFree( gameSceneMask );

		// Decode the bitmaps.
		for (uint n=1;n<gameNextScene;n++)
			anmDecode( n, incbin_scenes );

		anmDecode( gameNextScene+0, incbin_scenes );
		gameSceneBg = anmGetBitmap( WINDOW_WIDTH, WINDOW_HEIGHT );
		anmDecode( gameNextScene+1, incbin_scenes );
		gameSceneFg = anmGetBitmap( WINDOW_WIDTH, WINDOW_HEIGHT );
		anmDecode( gameNextScene+2, incbin_scenes );
		gameSceneMask = anmGetBitmap( WINDOW_WIDTH, WINDOW_HEIGHT );

		gamePrevScene = gameCurScene;
		gameCurScene = gameNextScene;
		command.exists = false;
		gameUiVisible = false;

		// Rebuild the pathing information.
		// First discover the clusters.
		memset( gamePathJoinX, 0, sizeof(gamePathJoinX) );
		memset( gamePathJoinY, 0, sizeof(gamePathJoinY) );
		memset( gamePathJoinN, 0, sizeof(gamePathJoinN) );
		memset( gamePathing, 0, sizeof(gamePathing) );
		for (uint j=0;j<16;j++)
			for (uint i=0;i<16;i++)
				gamePathDist[j][i] = 0xfff;
		for (uint j=0;j<16;j++)
			gamePathDist[j][j] = 0;

		for (uint y=0;y<gameSceneMask->h;y++)
		{
			for (uint x=1;x<gameSceneMask->w;x++)
			{
				u8 c1 = mask( x, y );
				if ( c1 > 16 )
					__debugbreak();

				for (int dy=-1;dy<=1;dy+=2)
				{
					for (int dx=-1;dx<=1;dx+=2)
					{
						u8 c0 = mask( x+dx, y+dy );
						if ( c0 != c1 && ( c0 && c1 ) )
						{
							gamePathJoinX[c0][c1] += x;
							gamePathJoinY[c0][c1] += y;
							gamePathJoinN[c0][c1] += 1;
							gamePathJoinX[c1][c0] += x+dx;
							gamePathJoinY[c1][c0] += y+dy;
							gamePathJoinN[c1][c0] += 1;

							gamePathDist[c0][c1] = 1;
							gamePathDist[c1][c0] = 1;
							gamePathing[c0][c1] = c1;
							gamePathing[c1][c0] = c0;
						}
					}
				}
			}
		}
		for (uint j=0;j<16;j++)
		{
			for (uint i=0;i<16;i++)
			{
				if ( gamePathJoinN[j][i] )
				{
					gamePathJoinX[j][i] /= gamePathJoinN[j][i];
					gamePathJoinY[j][i] /= gamePathJoinN[j][i];
				}
			}
		}

		// Now run the Floyd-Warshall algorithm to build links.
		for (uint k=0;k<16;k++)
			for (uint i=0;i<16;i++)
				for (uint j=0;j<16;j++)
					if ( gamePathDist[i][k] + gamePathDist[k][j] < gamePathDist[i][j] )
					{
						gamePathDist[i][j] = gamePathDist[i][k] + gamePathDist[k][j];
						gamePathing[i][j] = k;
					}

		scriptPrepareScene( gameCurScene );
	}

	// See if any command was issued.
	if ( mouseDown && ( gameHoverNoun.actor || gameHoverNoun.got || mouseY < WINDOW_HEIGHT ) )
	{
		if ( command.exists && command.needSecond && !command.hasSecond )
		{
			command.hasSecond = true;
			command.b = gameHoverNoun;
		} else {
			command.exists = true;
			command.needSecond = false;
			command.hasSecond = false;
			command.verb = gameVerb;
			command.a = gameHoverNoun;
			command.b.actor = NULL;
			command.b.got = NULL;
			command.x = mouseX;
			command.y = mouseY;
			gameClickTime = 0;
			gameHoldCommand = true;

			gameVerb = VERB_WALK; // reset the verb
		}
	}

	gameSchedule();

	if ( gameCurScene == 0 )
		return;

	// Build an actor list.
	uint numActors = 0;
	Actor *actors[MAX_ACTORS];
	for (uint j=0;j<MAX_ACTORS;j++)
	{
		if ( gameActors[j].valid )
			actors[numActors++] = &gameActors[j];
	}

	// Update the actors.
	for (uint n=0;n<numActors;n++)
		actorUpdate( actors[n] );

	// Sort the actors by Y
	for (uint j=0;j<numActors;j++)
	{
		uint best = 0;
		int bestY = 0x7fffffff;
		for (uint i=j;i<numActors;i++)
		{
			if ( actors[i]->y < bestY )
			{
				bestY = actors[i]->y;
				best = i;
			}
		}

		Actor *tmp = actors[best];
		actors[best] = actors[j];
		actors[j] = tmp;
	}
	
	// Build an inventory list.
	uint numInv = 0;
	Got *inv[MAX_INVENTORY];
	for (uint j=0;j<MAX_INVENTORY;j++)
	{
		if ( gameGot[j].name )
			inv[numInv++] = &gameGot[j];
	}

	// Sort the inventory by age
	for (uint j=0;j<numInv;j++)
	{
		uint best = 0;
		uint bestStamp = 0xffffffff;
		for (uint i=j;i<numInv;i++)
		{
			if ( inv[i]->timestamp < bestStamp )
			{
				bestStamp = inv[i]->timestamp;
				best = i;
			}
		}

		Got *tmp = inv[best];
		inv[best] = inv[j];
		inv[j] = tmp;
	}

	// See what's highlighted.
	gameHoverNoun.actor = NULL;
	for (uint n=0;n<numActors;n++)
	{
		Actor *actor = actors[n];
		if ( actor->name )
		{
			int w = actor->w;
			int h = actor->h;
			int hx = w/2, hy = h;
			if ( mouseX >= actor->x - hx && mouseX < actor->x - hx + w && mouseY >= actor->y - hy && mouseY < actor->y )
				gameHoverNoun.actor = actor;
		}
	}

	// Render the background
	lbmpBlit( hwScreen, gameSceneBg, 0, 0, 0, 0, gameSceneBg->w, gameSceneBg->h );

	// Render the actors.
	for (uint n=0;n<numActors;n++)
	{
		Actor *actor = actors[n];
		if ( !actor->costume )
			continue;
		
		// Find the most suitable frame.
		LBitmap *bmp = findBestFrame( actor->costume, actor->anim, actor->dir, actor->cycle );
		if ( bmp == NULL )
			bmp = findBestFrame( actor->costume, actor->anim, DIR_DOWN, actor->cycle );
		if ( bmp == NULL )
			bmp = findBestFrame( actor->costume, ANIM_IDLE, actor->dir, actor->cycle );
		if ( bmp == NULL )
			bmp = findBestFrame( actor->costume, ANIM_IDLE, DIR_DOWN, actor->cycle );
		if ( bmp == NULL )
			bmp = findBestFrame( actor->costume, ANIM_WALK, DIR_DOWN, actor->cycle );
		if ( bmp == NULL )
			__debugbreak();

		int hx = actor->costume->hotx;
		int hy = actor->costume->hoty;
		int w = actor->costume->sprW;
		int h = actor->costume->sprH;
		lbmpBlitMasked( hwScreen, bmp, actor->x-hx, actor->y-hy, 0, 0, w, h );

#ifdef DEBUG_ACTORS
		lbmpRect( hwScreen, actor->x-actor->w/2, actor->y-actor->h, actor->w, actor->h, COLOR_WHITE );
#endif
	}

	// Render the foreground
	lbmpBlitMasked( hwScreen, gameSceneFg, 0, 0, 0, 0, gameSceneFg->w, gameSceneFg->h );

	// Render the dialogue.
	for (uint n=0;n<numActors;n++)
	{
		Actor *actor = actors[n];
		if ( !actor->costume )
			continue;

		if ( actor->sayTicks > 0 )
		{
			actor->sayTicks--;
			if ( actor->sayTicks == 0 )
				actor->anim = ANIM_IDLE;

			fontPrint( actor->sayx, actor->sayy, actor->costume->color, actor->saying );
		}
	}

	// Render the textline
	lbmpRectFill( hwScreen, 0, WINDOW_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT-WINDOW_HEIGHT, COLOR_BLACK );
	if ( gameUiVisible )
	{
		u8 color = COLOR_LTGRAY;
		if ( command.exists || gameClickTime < 40 )
			color = COLOR_WHITE;

		if ( !command.exists )
			gameHoldCommand = false;

		if ( !gameHoldCommand )
		{
			const char *hover = getNounName( gameHoverNoun );
			if ( hover )
				wsprintf( gameCommandLine, "%s %s", verbs[gameVerb], hover );
			else
				wsprintf( gameCommandLine, "%s", verbs[gameVerb] );
		}

		if ( command.exists && command.needSecond && !command.hasSecond )
		{
			char *end = &gameCommandLine[lstrlen(gameCommandLine)];
			*end++ = ' ';

			const char *bit = "with";
			if ( command.verb == VERB_GIVE )
				bit = "to";

			const char *first = getNounName( command.a );
			const char *hover = getNounName( gameHoverNoun );
			if ( hover )
				wsprintf( gameCommandLine, "%s %s %s %s", verbs[command.verb], first, bit, hover );
			else
				wsprintf( gameCommandLine, "%s %s %s", verbs[command.verb], first, bit );
		}

		fontPrint( SCREEN_WIDTH/2, WINDOW_HEIGHT+1, color, gameCommandLine );
	}

	int guiY = WINDOW_HEIGHT + 10;
	if ( gameUiVisible )
	{
		// Render the GUI
		int x = 0;
		int y = guiY + 4;
		for (uint n=0;n<NVERBS;n++)
		{
			uint len = 0;
			while( verbs[n][len] )
				len++;

			u8 c = COLOR_LTGRAY;
			if ( mouseX >= 10+x && mouseY >= y && mouseX < 10+x+50 && mouseY < y+10 )
			{
				c = COLOR_WHITE;
				if ( mouseDown )
				{
					gameVerb = (Verb)n;
					gameHoldCommand = false;
				}
			}

			fontPrint2( 10 + x, y, c, verbs[n], len );
			x += 70;
			if ( x > 100 )
			{
				x = 0;
				y += 10;
			}
		}
	}

	if ( gameUiVisible )
	{
		// Render the inventory
		gameHoverNoun.got = NULL;

		uint idx = gameInvScroll;
		uint count = numInv - idx;
		if ( count > 10 )
			count = 10;
		
		int x = 200;
		int y = guiY + 4;
		while( count-- )
		{
			Got *got = inv[idx++];
			uint len = 0;
			while( got->name[len] )
				len++;


			if ( mouseX >= x && mouseX < 320 && mouseY >= y && mouseY < y+10 )
				gameHoverNoun.got = got;

			u8 c = COLOR_LTGRAY;
			if ( gameHoverNoun.got == got )
				c = COLOR_WHITE;

			fontPrint2( x, y, c, got->name, len );

			y += 10;
		}
	}

	// Update the conversation tree.
	if ( convNumOptions )
	{
		convSelected = -1;

		int x = 10;
		int y = guiY + 4;
		for (uint n=0;n<convNumOptions;n++)
		{
			const char *text = convOptions[n];
			uint len = 0;
			while( text[len] )
				len++;

			u8 c = COLOR_LTGRAY;
			if ( mouseX >= x && mouseX < 320 && mouseY >= y && mouseY < y+10 )
			{
				c = COLOR_WHITE;
				if ( mouseDown )
					convSelected = n;
			}

			fontPrint2( x, y, c, text, len );

			y += 10;
		}
	}
}

void wait( uint ticks )
{
	while( ticks-- )
		gameSchedule();
}

void waitwalk( Actor *actor )
{
	while( actor->anim == ANIM_WALK )
		gameSchedule();
}

void scriptWaitText( Actor *actor )
{
	while( actor->sayTicks > 0 )
		gameSchedule();
}

Got *scriptGive( const char *name, uint max, u32 flags )
{
	uint count = 0;
	Got *got = NULL;
	for (uint n=0;n<MAX_INVENTORY;n++)
	{
		if ( gameGot[n].name )
		{
			if ( !strcmp( gameGot[n].name, name ) )
				count++;
		} else {
			if ( !got )
				got = &gameGot[n];
		}
	}

	if ( count >= max )
		return NULL;

	if ( !got )
		__debugbreak();

	got->name = name;
	got->flags = flags;
	got->timestamp = gameTick;
	return got;
}

void scriptTake( const char *name )
{
	for (uint n=0;n<MAX_INVENTORY;n++)
	{
		if ( gameGot[n].name && !strcmp( name, gameGot[n].name ) )
		{
			gameGot[n].name = NULL;
			return;
		}
	}
}

Actor *scriptSpawn( const char *name, int x, int y, int w, int h, Costume *costume, u32 flags )
{
	int n;
	for (n=0;n<MAX_ACTORS;n++)
	{
		if ( !gameActors[n].valid )
			break;
	}

	if ( n == MAX_ACTORS )
		__debugbreak();

	if ( costume )
	{
		if ( w == 0 )
			w = costume->defW;
		if ( h == 0 )
			h = costume->defH;
	}

	Actor *actor = &gameActors[n];
	actor->name = name;
	actor->x = x;
	actor->y = y;
	actor->w = w;
	actor->h = h;
	actor->flags = flags;
	actor->tx = x;
	actor->ty = y;
	actor->anim = ANIM_IDLE;
	actor->dir = DIR_DOWN;
	actor->saying = "";
	actor->sayTicks = 0;
	actor->blockedTicks = 0;
	actor->costume = costume;
	actor->sayx = x;
	actor->sayy = y;
	actor->valid = true;

	static uint randish = 0;
	actor->cycle = randish++;

	return actor;
}

void scriptDelete( Actor *actor )
{
	actor->valid = false;
}

void fixTarget( Actor *actor, int curx, int cury, int &x, int &y )
{
	// Find a nice spot to stand near the object.
	int dist = 20 + actor->w/2;

	if ( curx < actor->x )
	{
		if ( x > actor->x - dist )
			x = actor->x - dist;
	} else {
		if ( x < actor->x + dist )
			x = actor->x + dist;
	}
	y = actor->y;
}

void walkto( Actor *actor, int x, int y )
{
	while( solid( x, y ) && y < WINDOW_HEIGHT )
		y++;

	actor->tx = x;
	actor->ty = y;
	actor->anim = ANIM_WALK;
	actor->blockedTicks = 0;
}

void say( Actor *actor, const char *text )
{
	actor->saying = text;
	if ( actor->anim != ANIM_WALK )
		actor->anim = ANIM_TALK;
	actor->sayx = actor->x;
	actor->sayy = actor->y;
	
	if ( actor->costume )
		actor->sayy -= actor->costume->hoty;
	actor->sayy -= 10;

	actor->sayTicks = 0;
	while( *text )
	{
		actor->sayTicks += 6;
		text++;
	}

	uint minTicks = 150;
	if ( actor->sayTicks < minTicks )
		actor->sayTicks = minTicks;
}

void saywait( Actor *actor, const char *text )
{
	say( actor, text );
	scriptWaitText( actor );
}


void lookat( Actor *actor, Actor *target )
{
	if ( target->x > actor->x )
		actor->dir = DIR_RIGHT;
	else
		actor->dir = DIR_LEFT;
}

void stop( Actor *actor )
{
	actor->anim = ANIM_IDLE;
	actor->tx = actor->x;
	actor->ty = actor->y;
}

bool standBy( Actor *actor, Actor *target )
{
	int tx = target->x, ty = target->y;
	if ( !( target->flags & FLAG_EXIT ) )
		fixTarget( target, actor->x, actor->y, tx, ty );
	
	if ( tx == actor->x && ty == actor->y )
	{
		actor->anim = ANIM_IDLE;
		lookat( actor, target );
		return true;
	}
	
	if ( actor->anim != ANIM_WALK )
		walkto( actor, tx, ty );
	
	// Update the target in case it moves.
	actor->tx = tx;
	actor->ty = ty;
	return false;
}

u32 getFlags( Noun noun )
{
	if ( noun.actor )
		return noun.actor->flags;
	if ( noun.got )
		return noun.got->flags;
	return 0;
}

bool has( const char *name )
{
	for (uint n=0;n<MAX_INVENTORY;n++)
	{
		if ( gameGot[n].name && !strcmp( name, gameGot[n].name ) )
			return true;
	}
	return false;
}

void updateGame()
{
	// Lock out the UI if we go into a modal loop.
	gameUiVisible = true;
	wait( 1 );
	gameUiVisible = false;
}

void conversation()
{
	convSelected = -1;
}

void options()
{
	wait( 1 );
	convNumOptions = 0;
}

bool said( u32 id )
{
	for (uint n=0;n<convNumSaid;n++)
	{
		if ( convSaid[n] == id )
			return true;
	}
	return false;
}

bool option( const char *text )
{
	uint idx = convNumOptions++;
	convOptions[idx] = text;

	if ( (int)idx == convSelected )
	{
		convNumOptions = 0;
		convSelected = -1;

		saywait( player, text );
		return true;
	}
	return false;
}

bool optiononce( u32 id, const char *text )
{
	if ( said( id ) )
		return false;

	if ( option( text ) )
	{
		convSaid[convNumSaid++] = id;
		return true;
	}

	return false;
}

bool operator==( Noun a, Noun b )
{
	if ( a.actor && b.actor && a.actor == b.actor )
		return true;
	if ( a.got && b.got && a.got == b.got )
		return true;
	return false;
}

bool operator==( Noun a, Actor *b )
{
	if ( a.actor && a.actor == b )
		return true;
	return false;
}

bool operator==( Noun a, Got *b )
{
	if ( a.got && a.got == b )
		return true;
	return false;
}
