#include "ludum.h"

// Some basic DPaint Animation loading code.
// Copied from something I Found On The Internet (tm).

struct AnmHeader
{
	u32 id;		/* 4 character ID == "LPF " */
	u16 maxLps;		/* max # largePages allowed. 256 FOR NOW.   */
	u16 nLps;		/* # largePages in this file. */
	u32 nRecords;	/* # records in this file.  65534 is current limit plus */
					/* one for last-to-first delta for looping the animation */
	u16 maxRecsPerLp;	/* # records permitted in an lp. 256 FOR NOW.   */
	u16 lpfTableOffset;	/* Absolute Seek position of lpfTable.  1280 FOR NOW.
							The lpf Table is an array of 256 large page structures
							that is used to facilitate finding records in an anim
							file without having to seek through all of the Large
							Pages to find which one a specific record lives in. */
	u32 contentType;	/* 4 character ID == "ANIM" */
	u16 width;		/* Width of screen in pixels. */
	u16 height;		/* Height of screen in pixels. */
	u8 variant;	/* 	0==ANIM. */
	u8 version;	/* 	0==frame rate is multiple of 18 cycles/sec.
					1==frame rate is multiple of 70 cycles/sec.  */
	u8 hasLastDelta;	/* 1==Last record is a delta from last-to-first frame. */
	u8 lastDeltaValid;	/* 0==The last-to-first delta (if present) hasn't been
							updated to match the current first&last frames,	so it
							should be ignored. */
	u8 pixelType;	/* 0==256 color. */
	u8 CompressionType;	/* 1==(RunSkipDump) Only one used FOR NOW. */
	u8 otherRecsPerFrm;	/* 0 FOR NOW. */
	u8 bitmaptype;  /* 	1==320x200, 256-color.  Only one implemented so far. */
	u8 recordTypes[32];	/* Not yet implemented. */
	u32 nFrames;	 /* 	In case future version adds other records at end of 
					 file, we still know how many actual frames.
					 NOTE: DOES include last-to-first delta when present. */
	u16 framesPerSecond;	/* Number of frames to play per second. */
	u16 pad2[29];	/* 58 u8s of filler to round up to 128 u8s total. */
};

/* this is the format of an large page structure */
struct AnmDescriptor {
	u16 baseRecord;	/* Number of first record in this large page. */
	u16 nRecords;	/* Number of records in lp.
					bit 15 of "nRecords" == "has continuation from previous lp".
					bit 14 of "nRecords" == "final record continues on next lp". */
	u16 nBytes;		/* Total number of u8s of contents, excluding header. */
};

AnmDescriptor *anmCurPage;
u8 anmScreen[0x10000];

/* given a frame number return the large page number it resides in */
u16 anmFindPage( u16 frame, uint numPages, AnmDescriptor *pages )
{
	for(u16 i=0;i<numPages;i++)
	{
		if ( pages[i].baseRecord <= frame && pages[i].baseRecord + pages[i].nRecords > frame )
			return i;
	}
	return 0;
}

/* seek out and load in the large page specified */
const u8 *anmGetPage( const u8 *file, u16 pagenumber )
{
	anmCurPage = (AnmDescriptor *)( file + 0x0B00 + (int)pagenumber * 0x10000 );

	u8 *ptr = (u8 *)( anmCurPage + 1 );
	return ptr + 2;
}

void anmDecompress( const u8 *srcP, u8 *dstP )
{
	u8 cnt;
	u16 wordCnt;
	u8 pixel;

nextOp:
	cnt = (signed char) *srcP++;
	if (cnt == 0)
		goto run;
	if (!(cnt & 0x80))
		goto dump;
	cnt &= 0x7f;
	if (cnt == 0)
		goto longOp;
	/* shortSkip */
	dstP += cnt;			/* adding 7-bit count to 32-bit pointer */
	goto nextOp;
dump:
	do
	{
		*dstP++ = *srcP++;
	} while (--cnt);

	dstP += cnt;
	srcP += cnt;
	goto nextOp;
run:
	wordCnt = (u8)*srcP++;		/* 8-bit unsigned count */
	pixel = *srcP++;
	do
	{
		*dstP++ = pixel;
	} while (--wordCnt);

	dstP += wordCnt;
	goto nextOp;
longOp:
	wordCnt = *((u16 *)srcP);
	srcP += 2;
	if ((s16)wordCnt <= 0)
		goto notLongSkip;	/* Do SIGNED test. */

	/* longSkip. */
	dstP += wordCnt;
	goto nextOp;

notLongSkip:
	if (wordCnt == 0)
		goto stop;
	wordCnt &= ~0x8000;		/* Remove sign bit. */
	if (wordCnt >= 0x4000)
		goto longRun;

	/* longDump. */
	do 
	{  
		*dstP++ = *srcP++;  
	} while (--wordCnt);

	dstP += wordCnt;
	srcP += wordCnt;
	goto nextOp;

longRun:
	wordCnt &= ~0x4000;		/* Clear "longRun" bit. */
	pixel = *srcP++;
	do 
	{  
		*dstP++ = pixel; 
	} while (--wordCnt);

	dstP += wordCnt;
	goto nextOp;

stop:	/* all done */
	;
}

void anmDecompressFrame( u16 frame, const u8 *pageData )
{
	u16 destframe = frame - anmCurPage->baseRecord;

	u16 *offsets = (u16 *)pageData;
	u16 offset = 0;
	for (u16 i=0;i<destframe;i++)
		offset += offsets[i];

	const u8 *ppointer = pageData + anmCurPage->nRecords*2 + offset;

	if ( ppointer[1] )
	{
		// Skip the extra data and keep it word aligned.
		u16 q = ((u16 *)ppointer)[1];
		ppointer += 4 + (q + (q & 1));
	} else {
		ppointer += 4;
	}

	anmDecompress( ppointer, anmScreen );
}

uint anmGetNumFrames( const u8 *data )
{
	AnmHeader *hdr = (AnmHeader *)data;
	if ( hdr->hasLastDelta )
		return hdr->nFrames - 1;
	else
		return hdr->nFrames;
}

void anmGetPalette( const u8 *data, u32 *palette )
{
	AnmHeader *hdr = (AnmHeader *)data;

	u8 *src = (u8 *)( data + sizeof(AnmHeader) + 128 );
	for(uint i=0;i<256;i++)
	{
		u8 b = *src++;;
		u8 g = *src++;;
		u8 r = *src++;;
		*src++;

		palette[i] = ( r << 16 ) | ( g << 8 ) | b;
	}
}

void anmDecode( uint frame, const u8 *data )
{
	AnmHeader *hdr = (AnmHeader *)data;
	frame--;

	u16 page = anmFindPage( frame, hdr->nLps, (AnmDescriptor *)( data + hdr->lpfTableOffset ) );
	const u8 *pageData = anmGetPage( data, page );
	anmDecompressFrame( frame, pageData );
}

LBitmap *anmGetBitmap( uint w, uint h )
{
	LBitmap *bmp = lbmpNew( w, h );
	for (uint y=0;y<h;y++)
		memcpy( &bmp->pixels[y*w], &anmScreen[y*320], w );

	return bmp;
}
