#!/usr/bin/perl -w
use Data::Dumper;
use File::Copy;

my @specs;
my $sdkdir = $ENV{KTSDK} . "\\biggle";
my $tmpdir = "tmp";
mkdir $sdkdir;
mkdir $tmpdir;
my @cfiles;

`p4checkout $sdkdir\\biggle.h`;
`p4checkout $sdkdir\\biggle.lib`;

die "Visual Studio environment is not set up.\n" unless $ENV{VSINSTALLDIR};

sub parseSpec
{
	my ($specfile, $prefix) = @_;
	open SPECFILE, $specfile or die "$specfile: $!\n";

	my $prevspec = undef;

	# Parse all specs in the spec file, into the @specs array.
	while(<SPECFILE>)
	{
		chomp;
		s/#.*$//;	# trim comments
		s/\s+$//;	# trim trailing whitespace
		
		next unless $_;
		
		next if /^\S+:/;	# skip properties
		
		my $line = $_;
		if ($line =~ /^\t/)
		{
			$line =~ s/^\t(\S+)// or die "Bad property line: $line";
			my $prop = $1, $value = "";
			$value = $1 if ($line =~ /\s+(\S+)/);
		
			$prevspec->{$prop} = $value;
		} else {
			push @specs, $prevspec if defined $prevspec;
			
			$line =~ /^(\S+)\(.*\)$/ or die "Bad spec def: $line";
			$prevspec = { def => ($prefix . $1) };
		}
	}
	
	close SPECFILE;
}

sub appendFile
{
	my ($filename) = @_;
	open INFILE, $filename or die "$filename: $!\n";

	print HFILE "\n/* --------- begin $filename ------------------------------------*/\n";
	while(<INFILE>)
	{
		print HFILE;
	}
	print HFILE "\n/* --------- end $filename --------------------------------------*/\n";
	close INFILE;
}

sub genCode
{
	my ($cfile, $hfile) = @_;

	open HFILE, ">$hfile" or die "$hfile: $!\n";
	
	# -------------- Build the list of functions ------------------
	my @funcs;
	foreach my $spec (@specs)
	{
		#print Dumper($spec);

		my $name = $spec->{def};
		my $cat = $spec->{category};
		
		next if $cat =~ /DEPRECATED$/;
		
		# Skip if built-in already.
		# Windows only supports up to 1.1 by default.
		next if $cat =~ /^VERSION_1_(0|1)/;
		
		push @funcs, $name;
	}	
	@funcs = sort @funcs;
	
	# ----------- Write the H file header ----------------
print HFILE <<'EOF';
/* Biggle - The least-big OpenGL extension wrapper library.
 * Our web page: http://www.codersnotes.com/programs/biggle
 *
 * Copyright (c) 2010, Richard Mitton. All rights reserved.
 *
 * I give you full permission to use this code for whatever
 * purpose you wish, in whatever form you wish. It would be
 * nice to put a credit somewhere in the documentation, but
 * it's up to you. Or send me free stuff. Free stuff rocks!
 *
 * Portions are copyright by The Khronos Group, however the
 * license for it allows everybody to use their code freely
 * without any legal issues, so there's no worries on that.
 *
 * To use biggle, include "biggle.h" from your source code,
 * and link with "biggle.lib". Finally call biggleInit() to
 * initialize all the extensions, although please make sure
 * that you have set up an OpenGL context already, else you
 * will encounter NULL-pointer exceptions and then sadness.
 */

#ifndef __BIGGLE_H__
#define __BIGGLE_H__

#ifdef __cplusplus
extern "C" {
#endif

#define BIGGLEAPI __stdcall

/* biggleInit() - Initializes all OpenGL extensions used in
 *                the program. Call this once you have made
 *                a valid OpenGL context. Calling it before
 *                will not work, as different contexts will
 *                sometimes not support all the extensions.
 */
void BIGGLEAPI biggleInit(void);

/* biggleSupported() - Returns 1 if this extension is valid
 *                     this card/context. Pass in an OpenGL
 *                     ARB-style name, such as GL_NV_fence.
 */
int BIGGLEAPI biggleSupported(const char *extension);


typedef void BIGGLEFN(void);

typedef struct
{
	BIGGLEFN **fn;
	const char *name;
} BIGGLEIMPORT;

#ifdef __cplusplus
}
#endif

/*-------------------------------------------------------*/
#ifndef APIENTRY
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>
#endif

#include <GL/gl.h>

EOF

	#----------- include the GL extensions headers in here ----
	appendFile("gl/glext.h");
	appendFile("gl/wglext.h");


	#------- declare all our Biggle functions -------------	
	print HFILE <<'EOF';

#ifdef __cplusplus
extern "C" {
#endif

EOF
	my $i = 0;
	foreach my $name (@funcs) {
		my $type = 'PFN' . uc($name) . 'PROC';
		print HFILE "extern $type $name;\n";
		$i++;
	}
	print HFILE <<'EOF';
#ifdef __cplusplus
}
#endif

#endif
EOF
	close HFILE;
	
	#--------- Write the stub files---------------------
	foreach my $name (@funcs)
	{
		my $stubfile = "$tmpdir/$name.c";
		open STUBFILE, ">$stubfile" or die "$stubfile: $!\n";
		print STUBFILE "#include \"biggle.h\"\n\n";
		
		my $type = 'PFN' . uc($name) . 'PROC';
		print STUBFILE "$type $name;\n\n";
		print STUBFILE "static const char name[] = \"$name\";\n\n";
		print STUBFILE "#pragma data_seg(\".biggle\$DATA\")\n\n";
		print STUBFILE "static BIGGLEIMPORT import = { (BIGGLEFN **)&$name, name };\n";
		close STUBFILE;
		
		push @cfiles, $stubfile;
	}
	
	# --- Write the main biggle C file -----------------------------
	open CFILE, ">$cfile" or die "$cfile: $!\n";
	print CFILE <<'EOF';
#include "biggle.h"

#pragma data_seg(".biggle$A")
static DWORD importStart = 0;
#pragma data_seg(".biggle$Z")
static DWORD importEnd = 0;
#pragma data_seg()
#pragma comment(linker, "/merge:.biggle=.data")

void BIGGLEAPI biggleInit(void)
{
	DWORD *ptr = (DWORD *)(&importStart + 1);
	while(ptr != (DWORD *)&importEnd)
	{
		BIGGLEIMPORT *import = (BIGGLEIMPORT *)ptr;
		if (import->fn)
		{
			*import->fn = (BIGGLEFN *)wglGetProcAddress(import->name);
			ptr++;
		}
		ptr++;
	}
}

int BIGGLEAPI biggleSupported(const char *extension)
{
	const char *str;
	int len, match;
	str = glGetString(GL_EXTENSIONS);
	while(*str)
	{
		match = 1;
		len = 0;
		while(*str && *str != ' ')
			match &= (*str++ == extension[len++]);
	
		if (match && extension[len] == 0)
			return 1;
			
		while(*str == ' ')
			str++;
	}
	
	return 0;
}
EOF
	
	close CFILE;
	push @cfiles, $cfile;
}

#----------- main ------------------------------------
parseSpec("gl/gl.spec", "gl");
parseSpec("gl/wglext.spec", "wgl");
genCode("$tmpdir/biggle.c", "$tmpdir/biggle.h");

my $libcmd = "$tmpdir\\lib.cmd";
open LIBCMD, ">$libcmd" or die "$libcmd: $!\n";

print LIBCMD "/nologo /OUT:$sdkdir\\biggle.lib";

foreach my $cfile (@cfiles)
{
	my $objfile = "$cfile.obj";
	system("cl.exe /nologo /Zl /Ox /I. /c $cfile /Fo$objfile") == 0 or die;
	print LIBCMD " $objfile";
}

close LIBCMD;
system("lib.exe \@$libcmd") == 0 or die;

copy("$tmpdir/biggle.h", "$sdkdir/biggle.h") or die;
